---
title: Multiple Cron Triggers · Cloudflare Workers docs
description: Set multiple Cron Triggers on three different schedules.
lastUpdated: 2025-08-18T14:27:42.000Z
chatbotDeprioritize: false
tags: Middleware,JavaScript,TypeScript
source_url:
  html: https://developers.cloudflare.com/workers/examples/multiple-cron-triggers/
  md: https://developers.cloudflare.com/workers/examples/multiple-cron-triggers/index.md
---

If you want to get started quickly, click on the button below.

[![Deploy to Cloudflare](https://deploy.workers.cloudflare.com/button)](https://deploy.workers.cloudflare.com/?url=https://github.com/cloudflare/docs-examples/tree/main/workers/multiple-cron-triggers)

This creates a repository in your GitHub account and deploys the application to Cloudflare Workers.

* JavaScript

  ```js
  export default {
    async scheduled(event, env, ctx) {
      // Write code for updating your API
      switch (event.cron) {
        case "*/3 * * * *":
          // Every three minutes
          await updateAPI();
          break;
        case "*/10 * * * *":
          // Every ten minutes
          await updateAPI2();
          break;
        case "*/45 * * * *":
          // Every forty-five minutes
          await updateAPI3();
          break;
      }
      console.log("cron processed");
    },
  };
  ```

* TypeScript

  ```ts
  interface Env {}
  export default {
    async scheduled(
      controller: ScheduledController,
      env: Env,
      ctx: ExecutionContext,
    ) {
      // Write code for updating your API
      switch (controller.cron) {
        case "*/3 * * * *":
          // Every three minutes
          await updateAPI();
          break;
        case "*/10 * * * *":
          // Every ten minutes
          await updateAPI2();
          break;
        case "*/45 * * * *":
          // Every forty-five minutes
          await updateAPI3();
          break;
      }
      console.log("cron processed");
    },
  };
  ```

* Hono

  ```ts
  import { Hono } from "hono";


  interface Env {}


  // Create Hono app
  const app = new Hono<{ Bindings: Env }>();


  // Regular routes for normal HTTP requests
  app.get("/", (c) => c.text("Multiple Cron Trigger Example"));


  // Export both the app and a scheduled function
  export default {
    // The Hono app handles regular HTTP requests
    fetch: app.fetch,


    // The scheduled function handles Cron triggers
    async scheduled(
      controller: ScheduledController,
      env: Env,
      ctx: ExecutionContext,
    ) {
      // Check which cron schedule triggered this execution
      switch (controller.cron) {
        case "*/3 * * * *":
          // Every three minutes
          await updateAPI();
          break;
        case "*/10 * * * *":
          // Every ten minutes
          await updateAPI2();
          break;
        case "*/45 * * * *":
          // Every forty-five minutes
          await updateAPI3();
          break;
      }
      console.log("cron processed");
    },
  };
  ```

## Test Cron Triggers using Wrangler

The recommended way of testing Cron Triggers is using Wrangler.

Cron Triggers can be tested using Wrangler by passing in the `--test-scheduled` flag to [`wrangler dev`](https://developers.cloudflare.com/workers/wrangler/commands/#dev). This will expose a `/__scheduled` (or `/cdn-cgi/handler/scheduled` for Python Workers) route which can be used to test using a HTTP request. To simulate different cron patterns, a `cron` query parameter can be passed in.

```sh
npx wrangler dev --test-scheduled


curl "http://localhost:8787/__scheduled?cron=*%2F3+*+*+*+*"


curl "http://localhost:8787/cdn-cgi/handler/scheduled?cron=*+*+*+*+*" # Python Workers
```
